package gui;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Desktop;
import java.awt.Dimension;
import java.awt.FlowLayout;
import java.awt.Font;
import java.awt.GridLayout;
import java.awt.Toolkit;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.File;
import javax.swing.BorderFactory;
import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JFileChooser;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTabbedPane;
import javax.swing.JTextField;
import javax.swing.JTextPane;
import javax.swing.border.EtchedBorder;
import javax.swing.border.TitledBorder;

import anaysis.PhysicalAnalyser;

/**
 * This class represents the main window of the application for physical units
 * checking in PEARL.
 * 
 * @author Christina K. Houben
 */
public class MainWindow extends JFrame {

	/** Version ID for serialisation. */
	private static final long serialVersionUID = 1L;
	
	/** Enumeration of file types for file choosing etc. */
	private enum FileType {PrlFileType, XmlFileType}
	
	/** Reference to the PEARL file in the file system. */
	private File PrlFile = null;
	
	/** Reference to the XML file in the file system. */
	private File XmlFile = null;
	
	/** Text field for the PRL filename. */
	private JTextField PrlField = new JTextField("*.PRL");
	
	/** Text field for the XML filename. */
	private JTextField XmlField = new JTextField("*.XML");
	
	/** Text area for formatted PEARL text. */
	private JTextPane PrlText = new JTextPane();
	
	/** Text area for formatted XML text. */
	private JTextPane XmlText = new JTextPane();
	
	/** Tabbed panel for showing input texts and output results. */
	private JTabbedPane tabular = new JTabbedPane(JTabbedPane.TOP);
	
	/** Builds and shows the main window of the application. */
	public MainWindow() {
		// title and closing operation
		setTitle("Static Physical Unit Analyser for PEARL");
		setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
		// customized size
		Dimension screen = Toolkit.getDefaultToolkit().getScreenSize();
		Dimension appdim = new Dimension(960, 600);
		setSize(appdim);
		setLocation(
				(int) (screen.getWidth() - appdim.getWidth()) / 2,
				(int) (screen.getHeight() - appdim.getHeight()) / 2);
		// minimum size
		setMinimumSize(new Dimension(560, 420));
		// graphical user interface
		createGui();
	}
	
	/** Creates all panels, buttons, labels and text fields. */
	private void createGui() {
		setLayout(new BorderLayout());
		setJMenuBar(createMenuBar());
		add(createLeftPanel(), BorderLayout.WEST);
		add(createCenterPanel(), BorderLayout.CENTER);
	}
	
	/** Creates the panel on the left side of the window. */
	private JPanel createLeftPanel() {
		JPanel panel = new JPanel(new FlowLayout());
		panel.setPreferredSize(new Dimension(250, 125));
		panel.add(createButtonPanel());
		panel.add(createInputPanel());
		panel.add(createCopyrightPanel());
		return panel;
	}
	
	/** Creates the subpanel for PRL and XML file name. */
	private JPanel createInputPanel() {
		JPanel panel = new JPanel(new GridLayout(0, 1));
		TitledBorder border = BorderFactory.createTitledBorder(
				BorderFactory.createEtchedBorder(EtchedBorder.LOWERED),
				"  Input Files  ");
		border.setTitleJustification(TitledBorder.CENTER);
		panel.setPreferredSize(new Dimension(250, 125));
		panel.setBorder(border);
		panel.add(new JLabel("PEARL filename:"));
		panel.add(createFileNamePanel(FileType.PrlFileType));
		panel.add(new JLabel("XML filename:"));
		panel.add(createFileNamePanel(FileType.XmlFileType));
		return panel;
	}
	
	/** Creates a text field and a button to show a file dialog. */ 
	private JPanel createFileNamePanel(final FileType type) {
		JPanel panel = new JPanel();
		panel.setLayout(new BorderLayout());
		// text component
		switch (type) {
		case PrlFileType:
			PrlField.setEditable(false);
			panel.add(PrlField, BorderLayout.CENTER);
			break;
		case XmlFileType:
			XmlField.setEditable(false);
			panel.add(XmlField, BorderLayout.CENTER);
			break;
		}
		// button component
		JButton button = new JButton("...");
		button.setPreferredSize(new Dimension(32, 32));
		button.setToolTipText("browse ...");
		panel.add(button, BorderLayout.EAST);
		// button action
		switch (type) {
		case PrlFileType:
			button.addActionListener(new PRLFileListener());
			break;
		case XmlFileType:
			button.addActionListener(new XMLFileListener());
			break;
		}
		return panel;
	}
	
	/** Action listener for loading a PEARL file. */
	private class PRLFileListener implements ActionListener {
		public void actionPerformed(ActionEvent e) {
			JFileChooser chooser = new JFileChooser();
			chooser.setCurrentDirectory(new File("input"));
			int response = chooser.showOpenDialog(null);
			if (response == JFileChooser.APPROVE_OPTION) {
				PrlFile = chooser.getSelectedFile();
				PrlField.setText(chooser.getSelectedFile().getName());
				PrlText.setText(StyledFileReader.readTextFile(PrlFile));
				PrlText.setCaretPosition(0);
				tabular.setSelectedIndex(0);
			}
		}
	}
	
	/** Action listener for loading an XML file. */
	private class XMLFileListener implements ActionListener {
		public void actionPerformed(ActionEvent e) {
			JFileChooser chooser = new JFileChooser();
			chooser.setCurrentDirectory(new File("input"));
			int response = chooser.showOpenDialog(null);
			if (response == JFileChooser.APPROVE_OPTION) {
				XmlFile = chooser.getSelectedFile();
				XmlField.setText(chooser.getSelectedFile().getName());
				XmlText.setText(StyledFileReader.readTextFile(XmlFile));
				XmlText.setCaretPosition(0);
				tabular.setSelectedIndex(1);
			}
		}
	}
	
	/** Creates the subpanel for PRL and XML file name. */
	private JPanel createButtonPanel() {
		JPanel panel = new JPanel(new FlowLayout(FlowLayout.CENTER, 5, 1));
		panel.setPreferredSize(new Dimension(250, 50));
		panel.add(createSpeedButton(
				"recycle.png", "clear all",	new RecycleListener()));
		panel.add(createSpeedButton(
				"measure.png", "static analysis", new AnalysisListener()));
		panel.add(createSpeedButton(
				"question.png", "instruction manual", new ManualListener()));
		return panel;
	}
	
	/** Creates the subpanel for the copyright information. */
	private JPanel createCopyrightPanel() {
		JPanel panel = new JPanel(new GridLayout(0, 1));
		TitledBorder border = BorderFactory.createTitledBorder(
				BorderFactory.createEtchedBorder(EtchedBorder.LOWERED),
				"  Copyright Information  ");
		border.setTitleJustification(TitledBorder.CENTER);
		panel.setPreferredSize(new Dimension(250, 150));
		panel.setBorder(border);
		final String gap = "                 ";
		panel.add(new JLabel("    Java Implementation:"));
		panel.add(new JLabel(gap + "(C) 2015 Christina K. Houben"));
		panel.add(new JLabel(gap + "Chair of Computer Engineering"));
		panel.add(new JLabel(gap + "FernUniversitt in Hagen"));
		panel.add(new JLabel("    Current Version:"));
		JLabel label1 = new JLabel(gap + "This is work in progress,");
		JLabel label2 = new JLabel(gap + "final version will come soon.");
		label1.setBackground(Color.RED);
		label2.setBackground(Color.RED);
		label1.setOpaque(true);
		label2.setOpaque(true);
		panel.add(label1);
		panel.add(label2);
		return panel;
	}
	
	/** Creates the menu bar of the main window. */
	private JMenuBar createMenuBar() {
		JMenuBar bar = new JMenuBar();
		JMenu menu1 = new JMenu("File");
		JMenuItem item1 = new JMenuItem("Open PEARL file ...");
		JMenuItem item2 = new JMenuItem("Open XML file ...");
		JMenuItem item3 = new JMenuItem("Exit");
		item1.addActionListener(new PRLFileListener());
		item2.addActionListener(new XMLFileListener());
		item3.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent arg0) {
				System.exit(0);
			}
		});
		menu1.add(item1);
		menu1.add(item2);
		menu1.addSeparator();
		menu1.add(item3);
		bar.add(menu1);
		return bar;
	}
	
	/** Creates a speed button with an icon, a tool tip text and an action. */
	private JButton createSpeedButton(String filename, String tooltip,
			ActionListener listener) {
		JButton button = new JButton(new ImageIcon("dat/" + filename));
		button.setPreferredSize(new Dimension(40, 40));
		button.setToolTipText(tooltip);
		button.addActionListener(listener);
		return button;
	}
	
	/** Action listener for the recycle resp. clear all button. */
	private class RecycleListener implements ActionListener {
		public void actionPerformed(ActionEvent e) {
			PrlFile = null;
			XmlFile = null;
			PrlField.setText("*.PRL");
			XmlField.setText("*.XML");
			PrlText.setText("");
			XmlText.setText("");
		}
	}
	
	/** Action listener to start the static analysis. */
	private class AnalysisListener implements ActionListener {
		public void actionPerformed(ActionEvent e) {
			new PhysicalAnalyser(PrlText.getText(), XmlText.getText());
			tabular.setSelectedIndex(2);
		}
	}
	
	/** Action listener to show the instruction manual. */
	private class ManualListener implements ActionListener {
		public void actionPerformed(ActionEvent e) {
			try {
				Desktop.getDesktop().open(
						new File("dat/instruction-manual-german.pdf"));
			}
			catch (Exception x) {
				JOptionPane.showMessageDialog(
						null,
						"Error while opening instruction-manual-german.pdf!",
						"Instruction Manual",
						JOptionPane.ERROR_MESSAGE);
			}
		}
	}
	
	/** Creates the tabbed panel on the right of the main window. */
	private JTabbedPane createCenterPanel() {
		tabular.addTab("PEARL File", createPEARLPanel());
		tabular.addTab("XML File", createXMLPanel());
		tabular.addTab("Analysis Output", createOutputPanel());
		return tabular;
	}
	
	/** Creates the panel that shows the PEARL source text. */
	private JComponent createPEARLPanel() {
		PrlText.setEditable(false);
		PrlText.setFont(new Font("monospaced", Font.PLAIN, 12));
		JScrollPane panel = new JScrollPane(PrlText);
		return panel;
	}
	
	/** Creates the panel that shows the XML source text. */
	private JComponent createXMLPanel() {
		XmlText.setEditable(false);
		XmlText.setFont(new Font("monospaced", Font.PLAIN, 12));
		JScrollPane panel = new JScrollPane(XmlText);
		return panel;
	}
	
	/** Creates the panel that shows ste physical unit checking output. */
	private JPanel createOutputPanel() {
		JPanel panel = new JPanel();
		return panel;
	}
	
}
